# -*- coding: utf-8 -*-
"""
@author: tobias.bosse@de.bosch.com
"""
import sys
import os
import threading
from pprint import pprint
import time
import configparser
import uuid

##Openapi Integration
import openapi_client_strategy
import openapi_client_ai
import openapi_client_sim

# Configure the Server
import connexion
from openapi_server_strategy import encoder

# Read API keys and ports from the .ini file
config = configparser.ConfigParser()
config.read(os.path.join(os.path.dirname(__file__), "api_config.ini"))

# Configure the clients
# Strategy
strategy_configuration = openapi_client_strategy.Configuration(
    host=f"http://127.0.0.1:{config.get('PORTS', 'strategy_port')}/aissi/strategy/adapter/api/v1"
)
strategy_configuration.api_key["ApiKeyAuth"] = config.get("API_KEYS", "strategy_key")

# AI
ai_configuration = openapi_client_ai.Configuration(
    host=f"http://127.0.0.1:{config.get('PORTS', 'ai_port')}/aissi/ai/adapter/api/v1"
)
ai_configuration.api_key["ApiKeyAuth"] = config.get("API_KEYS", "ai_key")

# Sim
sim_configuration = openapi_client_sim.Configuration(
    host=f"http://127.0.0.1:{config.get('PORTS', 'sim_port')}/aissi/simulation/adapter/api/v1"
)
sim_configuration.api_key["ApiKeyAuth"] = config.get("API_KEYS", "sim_key")


def run_server():
    app = connexion.App(
        __name__,
        specification_dir="./Generator/server_strategy/openapi_server_strategy/openapi/",
    )
    app.app.json_encoder = encoder.JSONEncoder
    app.add_api(
        "openapi.yaml",
        arguments={"title": "AISSI Platform Strategy Adapter Service API"},
        pythonic_params=True,
    )
    app.run(port=config.get("PORTS", "strategy_port"))


def begin():
    """
    Starts a server thread and returns the thread object.

    Returns:
        threading.Thread: The server thread object.
    """
    server_thread = threading.Thread(target=run_server, daemon=True)
    server_thread.start()
    time.sleep(1)
    return server_thread


def idle(server_thread):
    """
    Monitors the status of the server thread and prints a message indicating whether it is running or stopped.

    Args:
        server_thread (Thread): The server thread to monitor.

    Raises:
        KeyboardInterrupt: If the user interrupts the program.

    Returns:
        None
    """
    try:
        while True:
            if server_thread.is_alive():
                pass
                # print("Server thread is still running.")
            else:
                print("Server thread has stopped.")
                break
            time.sleep(5)  # Check every 5 seconds
    except KeyboardInterrupt:
        # server_thread.stop() #thread stops automatically because it is daemon
        print("Server is shutting down...")
        sys.exit()


def createAgent(dispatchstrategy):
    """
    Create an agent with the specified dispatch strategy.

    Args:
        dispatchstrategy: The dispatch strategy for the agent.

    Returns:
        None
    """
    print(f"requesting agent with strategy {dispatchstrategy}")
    from openapi_client_ai.rest import ApiException

    request_ai_create_ai_model_message = (
        openapi_client_ai.RequestAICreateAIModelMessage(
            id=str(uuid.uuid4()),
            senderType="STRATEGY",
            interactionType="REQUEST",
            aiConfiguration=dispatchstrategy,
        )
    )
    with openapi_client_ai.ApiClient(ai_configuration) as api_client:
        api_instance = openapi_client_ai.AIModelManagementRequestApi(api_client)
        try:
            api_response = api_instance.request_create_ai_model(
                request_ai_create_ai_model_message
            )
            pprint(api_response)

        except ApiException as e:
            print(
                "Exception when calling RequestAiCreateAiModelMessage->request_create_ai_model: %s\n"
                % e
            )


def createFab(type):
    if type == "MiniFab":
        createMiniFab()
    else:
        print("Unknown Fab type")


def createMiniFab():
    """
    Creates a Fab environment of the specified type.

    Args:
        FabType (str): The type of Fab environment to create.

    Returns:
        None
    """
    print(f"requesting Minifab environment")
    from openapi_client_sim.rest import ApiException

    request_sim_create_env_message = (
        openapi_client_sim.RequestSimulationCreateEnvironmentMessage(
            id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
            senderId="3fa85f64-5717-4562-b3fc-2c963f66afa6",
            senderType="STRATEGY",
            senderVersion="string",
            interactionId="3fa85f64-5717-4562-b3fc-2c963f66afa1",
            interactionType="REQUEST",
            interactionMethod="string",
            rollOutDuration=864000,
            stepDuration=86400,
            reportInterval=86400,
            seed=666,
            data_set_id=1704668101358,
            kpi_definition=openapi_client_sim.KpiDefinition(
                id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                environmentId="00000000-0000-0000-0000-000000000000",
                kpiDefinitionItems=[
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="COMPLETED_WAFERS",
                        aggregationLevel="FAB",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="MASK_LAYER_PER_DAY",
                        aggregationLevel="FAB",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="COMPLETED_LAYERS",
                        aggregationLevel="FAB",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="TOTAL_TARDINESS",
                        aggregationLevel="FAB",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="TOTAL_STAGE_TARDINESS",
                        aggregationLevel="FAB",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="TOTAL_WAFER_MOVES",
                        aggregationLevel="FAB",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="COMPLETED_WAFERS",
                        aggregationLevel="PRODUCT",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="MASK_LAYER_PER_DAY",
                        aggregationLevel="PRODUCT",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="COMPLETED_LAYERS",
                        aggregationLevel="PRODUCT",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="TOTAL_TARDINESS",
                        aggregationLevel="PRODUCT",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="TOTAL_STAGE_TARDINESS",
                        aggregationLevel="PRODUCT",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="TOTAL_WAFER_MOVES",
                        aggregationLevel="PRODUCT",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="TOTAL_WAFER_MOVES",
                        aggregationLevel="EQUIPMENT_GROUP",
                        levelName="",
                    ),
                    openapi_client_sim.KpiDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        kpiName="TOTAL_WAFER_MOVES",
                        aggregationLevel="EQUIPMENT",
                        levelName="",
                    ),
                ],
            ),
            state_definition=openapi_client_sim.StateDefinition(
                id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                environmentId="00000000-0000-0000-0000-000000000000",
                stateDefinitionItems=[
                    openapi_client_sim.StateDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        dataEntityName="PRODUCTS",
                        dataEntityCategory="STATIC",
                    ),
                    openapi_client_sim.StateDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        dataEntityName="ROUTES",
                        dataEntityCategory="STATIC",
                    ),
                    openapi_client_sim.StateDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        dataEntityName="ROUTE_STAGE_PRELEAD_TIME",
                        dataEntityCategory="STATIC",
                    ),
                    openapi_client_sim.StateDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        dataEntityName="EQUIPMENTS",
                        dataEntityCategory="STATIC",
                    ),
                    openapi_client_sim.StateDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        dataEntityName="DEDICATIONS",
                        dataEntityCategory="STATIC",
                    ),
                    openapi_client_sim.StateDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        dataEntityName="PROCESS_PARAMS",
                        dataEntityCategory="STATIC",
                    ),
                    openapi_client_sim.StateDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        dataEntityName="CHAMBER_MAPPINGS",
                        dataEntityCategory="STATIC",
                    ),
                    openapi_client_sim.StateDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        dataEntityName="CHAMBER_DEDICATION",
                        dataEntityCategory="STATIC",
                    ),
                    openapi_client_sim.StateDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        dataEntityName="WIP",
                        dataEntityCategory="DYNAMIC",
                    ),
                    openapi_client_sim.StateDefinitionItem(
                        id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                        dataEntityName="EQUIPMENT_STATE",
                        dataEntityCategory="DYNAMIC",
                    ),
                ],
            ),
            action_definition=openapi_client_sim.ActionDefinition(
                id="3fa85f64-5717-4562-b3fc-2c963f66afa6",
                environmentId="00000000-0000-0000-0000-000000000000",
                actionCategory="LOT_DISPATCHING",
            ),
        )
    )

    with openapi_client_sim.ApiClient(sim_configuration) as api_client:
        api_instance = openapi_client_sim.SimulationManagementRequestApi(api_client)
        try:
            api_response = api_instance.request_create_environment(
                request_sim_create_env_message
            )
            pprint(api_response)
        except ApiException as e:
            print(
                "Exception when calling SimulationManagementRequestApi->request_create_environment: %s\n"
                % e
            )


def startSimulation(env_index, agent_index):
    """
    Starts a simulation with the specified environment and agent.

    Args:
        env_index (int): The index of the environment.
        agent_index (int): The index of the agent.

    Returns:
        None
    """
    print(f"Starting Simulation {env_index} with agent {agent_index}")
    from openapi_client_sim.rest import ApiException

    request_sim_rollout_env_message = (
        openapi_client_sim.RequestSimulationRunRolloutMessage(
            id=str(uuid.uuid4()),
            senderType="STRATEGY",
            interactionType="REQUEST",
            environmentId=str(env_index),
            aiModelId=str(agent_index),
        )
    )
    with openapi_client_sim.ApiClient(sim_configuration) as api_client:
        api_instance = openapi_client_sim.SimulationRequestApi(api_client)
        try:
            api_response = api_instance.request_run_rollout(
                request_sim_rollout_env_message
            )
            pprint(api_response)

        except ApiException as e:
            print(
                "Exception when calling SimulationManagementRequestApi->request_create_environment: %s\n"
                % e
            )


# Overriding of Server functions


from openapi_server_strategy.models.response_strategy_create_environment_message import (
    ResponseStrategyCreateEnvironmentMessage,
)


def edited_response_create_environment(body):
    if connexion.request.is_json:
        response_strategy_create_environment_message = (
            ResponseStrategyCreateEnvironmentMessage.from_dict(
                connexion.request.get_json()
            )
        )  # noqa: E501
    env_id = response_strategy_create_environment_message.environment_id
    print(f"Simulation got id {env_id}")
    global env_index
    env_index = env_id
    return "Ok ,ID received"


from openapi_server_strategy.controllers import strategy_response_controller

strategy_response_controller.response_create_environment = (
    edited_response_create_environment
)


from openapi_server_strategy.models.response_strategy_create_ai_model_message import (
    ResponseStrategyCreateAIModelMessage,
)


def edited_response_create_ai_model(body):
    if connexion.request.is_json:
        response_strategy_create_ai_message = (
            ResponseStrategyCreateAIModelMessage.from_dict(connexion.request.get_json())
        )  # noqa: E501
    ai_id = response_strategy_create_ai_message.ai_model_id
    print(f"Agent got id {ai_id}")
    global agent_index
    agent_index = ai_id
    return "Ok, ID received"


from openapi_server_strategy.controllers import strategy_response_controller

strategy_response_controller.response_create_ai_model = edited_response_create_ai_model


from openapi_server_strategy.models.response_strategy_run_rollout_message import (
    ResponseStrategyRunRolloutMessage,
)


def edited_response_roll_out(body):
    if connexion.request.is_json:
        response_strategy_roll_out_message = (
            ResponseStrategyRunRolloutMessage.from_dict(connexion.request.get_json())
        )  # noqa: E501)
    if response_strategy_roll_out_message.reward is None:
        print("KPI of dummy Rollout received")
    else:
        print("Factory KPIs:")
        kpi_values = {}
        for item in response_strategy_roll_out_message.reward.kpi_data_items:
            if item.aggregation_level == "FAB":
                if item.kpi_name not in kpi_values:
                    kpi_values[item.kpi_name] = []
                kpi_values[item.kpi_name].append(item.value)

        for kpi_name, values in kpi_values.items():
            aggregated_value = sum(values)
            print(f"{kpi_name}: {aggregated_value}")

    return "Ok, Roll out Results received"


from openapi_server_strategy.controllers import strategy_response_controller

strategy_response_controller.response_run_rollout = edited_response_roll_out


env_index = None
agent_index = None

if __name__ == "__main__":
    server_thread = begin()
    createAgent("FIFO")
    createFab("MiniFab")
    print("wait for response")  # wait for response of Simulation and AI
    counter = 0
    while env_index == None or agent_index == None:
        time.sleep(0.01)
        counter += 0.01
        print(f"\033[90m\rCounter: {counter:.3f} seconds\033[0m", end="")
    print("\nStart Roll Out")
    startSimulation(env_index, agent_index)
    idle(server_thread)
