# -*- coding: utf-8 -*-
"""
@author: tobias.bosse@de.bosch.com
"""

import os
from pprint import pprint
import configparser
import threading
import uuid

##Openapi Integration
import openapi_client_strategy
import openapi_client_ai
import openapi_client_sim


# Configure the Server
import connexion
from openapi_server_ai import encoder

# Read API keys and ports from the .ini file
config = configparser.ConfigParser()
config.read(os.path.join(os.path.dirname(__file__), "api_config.ini"))

# Configure the clients
# Strategy
strategy_configuration = openapi_client_strategy.Configuration(
    host=f"http://127.0.0.1:{config.get('PORTS', 'strategy_port')}/aissi/strategy/adapter/api/v1"
)
strategy_configuration.api_key["ApiKeyAuth"] = config.get("API_KEYS", "strategy_key")

# AI
ai_configuration = openapi_client_ai.Configuration(
    host=f"http://127.0.0.1:{config.get('PORTS', 'ai_port')}/aissi/ai/adapter/api/v1"
)
ai_configuration.api_key["ApiKeyAuth"] = config.get("API_KEYS", "ai_key")

# Sim
sim_configuration = openapi_client_sim.Configuration(
    host=f"http://127.0.0.1:{config.get('PORTS', 'sim_port')}/aissi/simulation/adapter/api/v1"
)
sim_configuration.api_key["ApiKeyAuth"] = config.get("API_KEYS", "sim_key")


def run_server():
    app = connexion.App(
        __name__, specification_dir="./Generator/server_ai/openapi_server_ai/openapi/"
    )
    app.app.json_encoder = encoder.JSONEncoder
    app.add_api(
        "openapi.yaml",
        arguments={"title": "AISSI Platform AI Adapter Service API"},
        pythonic_params=True,
    )
    app.run(port=config.get("PORTS", "ai_port"))


class AgentCollection:
    def __init__(self):
        self.agent_dict = {}
        self.interactions_dict = {}

    def add_dispatcher(self, obj):
        obj_id = uuid.uuid4()
        self.agent_dict[obj_id] = obj
        return obj_id

    def get_dispatch_function(self, index):
        return (
            self.agent_dict[index].dispatch_func if index in self.agent_dict else None
        )


class Dispatcher(object):
    def __init__(self, dispatch_func):
        self.dispatch_func = dispatch_func

    def dispatch(self, *args, **kwargs):
        return self.dispatch_func(*args, **kwargs)


def FIFO_Scores(request_message):
    current_time = request_message.state.current_date_time
    wip_list = request_message.state.wip
    dispatchingDecisions = []
    for wip in wip_list:
        lot_name = wip.lot
        arrival_time = wip.enter_operation_time
        score = (current_time - arrival_time).total_seconds()  # transform to seconds
        decision = openapi_client_sim.DispatchingDecision(
            lot=lot_name,
            route=wip.route,
            recipe=wip.recipe,
            mainStage=wip.main_stage,
            step=wip.step,
            subStep=wip.sub_step,
            score=score,
            reworkRoute=wip.rework_route,
            reworkStep=wip.rework_step,
            reworkSubStep=wip.rework_sub_step,
        )
        dispatchingDecisions.append(decision)
    sendScores(dispatchingDecisions, request_message)


def startAgentFIFO():
    global AICollection
    FIFO_dispatcher = Dispatcher(dispatch_func=FIFO_Scores)  # Create Dispatcher
    agent_id = AICollection.add_dispatcher(FIFO_dispatcher)
    print(f"Simple FIFO dispatcher created with ID:{agent_id}")
    sendIndex(agent_id)


def sendIndex(agent_id):
    from openapi_client_strategy import ApiException

    response_strategy_create_ai_message = (
        openapi_client_strategy.ResponseStrategyCreateAIModelMessage(
            id=str(uuid.uuid4()),
            senderType="AI",
            interactionType="RESPONSE",
            aiModelId=str(agent_id),
        )
    )
    with openapi_client_strategy.ApiClient(strategy_configuration) as api_client:
        api_instance = openapi_client_strategy.StrategyResponseApi(api_client)
        try:
            api_response = api_instance.response_create_ai_model(
                response_strategy_create_ai_message
            )
            pprint(api_response)
        except ApiException as e:
            print(
                "Exception when calling response_strategy_create_environment_message->response_create_ai_model: %s\n"
                % e
            )


def sendScores(dispatchingDecisions, request_ai_optimal_action_message):
    print("send scores")
    from openapi_client_sim import ApiException

    response_simulation_response_action_message = (
        openapi_client_sim.ResponseSimulationComputeOptimalActionMessage(
            id=str(uuid.uuid4()),
            senderId=request_ai_optimal_action_message.sender_id,
            enderType="AI",
            interactionId=request_ai_optimal_action_message.interaction_id,
            interactionType="RESPONSE",
            interactionMethod=request_ai_optimal_action_message.interaction_method,
            dispatch_action=openapi_client_sim.DispatchAction(
                id=str(uuid.uuid4()),
                environmentId=request_ai_optimal_action_message.state.environment_id,
                stateId=request_ai_optimal_action_message.state.id,
                dispatchingDecisions=dispatchingDecisions,
            ),
        )
    )
    with openapi_client_sim.ApiClient(sim_configuration) as api_client:
        api_instance = openapi_client_sim.SimulationResponseApi(api_client)
        try:
            api_response = api_instance.response_compute_optimal_action(
                response_simulation_response_action_message
            )
            pprint(api_response)
        except ApiException as e:
            print(
                "Exception when calling response_simulation_response_action_message->response_optimal_action: %s\n"
                % e
            )


from openapi_server_ai.models.request_ai_create_ai_model_message import (
    RequestAICreateAIModelMessage,
)
from openapi_server_ai.models.request_ai_compute_optimal_action_message import (
    RequestAIComputeOptimalActionMessage,
)


def edited_request_create_ai_model():
    """Create an AI model

    Create a new AI model. Receive the request id and the AI model configuration. Return the AI model id

    :param request_ai_create_ai_model_message: request creation an AI model
    :type request_ai_create_ai_model_message: dict | bytes

    :rtype: Union[str, Tuple[str, int], Tuple[str, int, Dict[str, str]]
    """
    if connexion.request.is_json:
        request_ai_create_ai_model_message = RequestAICreateAIModelMessage.from_dict(
            connexion.request.get_json()
        )  # noqa: E501

    if request_ai_create_ai_model_message.ai_configuration == "FIFO":
        threading.Thread(target=startAgentFIFO).start()
        return "OK, create Agent"
    else:
        return "I do not know this dispatching strategy yet!"


from openapi_server_ai.controllers import ai_model_management_request_controller

ai_model_management_request_controller.request_create_ai_model = (
    edited_request_create_ai_model
)


def edited_request_compute_action():
    """Compute an action

    Compute an optimal action of an AI in the given state. Receive the request id, the AI model id, and the state. Return the request id and the action. # noqa: E501

    :param request_ai_optimal_action_message: request an optimal action
    :type request_ai_optimal_action_message: dict | bytes

    :rtype: Union[str, Tuple[str, int], Tuple[str, int, Dict[str, str]]
    """
    if connexion.request.is_json:
        request_ai_optimal_action_message = (
            RequestAIComputeOptimalActionMessage.from_dict(connexion.request.get_json())
        )  # noqa: E501

    agent_id = uuid.UUID(request_ai_optimal_action_message.ai_model_id)
    threading.Thread(
        target=AICollection.get_dispatch_function(agent_id),
        args=(request_ai_optimal_action_message,),
    ).start()

    return "calculating scores..."


from openapi_server_ai.controllers import ai_request_controller

ai_request_controller.request_compute_optimal_action = edited_request_compute_action


AICollection = AgentCollection()


if __name__ == "__main__":
    run_server()
